/* vim: set wm=0 autowrite:
 *
 * wirrel.c -- Builds Happy Cubes (Wirrel Warrels)
 * Copyright (C) 1992-2002 Thomer M. Gil
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Thomer M. Gil (e-mail: [my first name]@[my last name].com)
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define VERSION                 "1.5"

#define MAX_WIR_NAME_LEN       32	// max name of wirrel
#define NR_WIRS               180	// max # of wirrels
#define NBITS                  16	// # of bits on wirrel
#define BIT_SPECS               5	// how much info to ...Fit()
#define COUNT_STOP        1000000       // how often report?
#define NR_EQUALS              16	// how many equals in a set pieces?

#define WIR_DES_FILE            "des/wirlist.des"
#define WIR_MAKE_FILE           "mak/wir222.mak"
#define MAX_FILE_LEN            32	/* Max name of file */

#define BIT(n)  ((structure[FitBits[n].piece]->bit[structure[FitBits[n].piece]->pos] >> FitBits[n].bit) & 1)

/* Description of piece */
struct piece_t {
  char name[MAX_WIR_NAME_LEN];
  short bit[8];                 // bit pattern for all positions
  char symm[8];		        // symm[n] == m means pos n equal to pos m
  signed equals[NR_EQUALS];	// equals[] == m means this piece equals m
  bool busy;		        // is used in structure
  unsigned nequals;	        // How many equal pieces?
  char pos;		        // index into bit[]
};

/* This struct narrows down to one bit on one piece */
struct bit_t {
  short piece;                  /* What piece */
  char bit;                     /* What bit */
};

/* This struct contains a function pointer and an array with piece- and bit
 * numbers.
 */
struct recipe_t {
  bool valid;
  int (*Fits) (struct bit_t FitBits[BIT_SPECS],	/* Function to call */
	       struct piece_t * structure[NR_WIRS]);
  struct bit_t FitBits[BIT_SPECS];	/* Parameter to function */
};

unsigned nsolutions = 0;	// solutions found so far
unsigned ncomplete = 0;	        // number of pieces in complete structure
unsigned navailable = 0;	// number of available pieces
unsigned long int tries = 0;	// total combinations tried
unsigned counter = 1;	        //
int errno;
unsigned verbose = 0;           // verbosity
unsigned silence = 0;           // solution silence
char docount = 0;               // whether or not to count

struct piece_t pieces_list[NR_WIRS];
struct recipe_t recipe[NR_WIRS][NBITS];


void
usage(char *argv[])
{
  printf("\
Wirrel %s\n\
\n\
Usage: %s [OPTION]...\n\
\n\
options:\n\
  -h, --help                         show this help\n\
  -d, --description file             file contains piece descriptions\n\
  -m, --recipe_file file             file contains build recipe\n\
  -s, --silent                       continue after giving solution\n\
  -c, --count                        give count information\n\
  -v, --verbose                      add verbosity\n", VERSION, argv[0]);
}


void print_bits(short bits)
{
  for (unsigned i = 0; i < NBITS; i++)
    printf("%d", (bits & (1 << i)) != 0);
}


short
flip(short bits, unsigned position)
{
  if (verbose >= 3) {
    printf("Flipping from pos %d: ", position);
    print_bits(bits);
    printf("\n");
  }

  // turn right by shifting bits to left
  if (position != 3 && position != 7)
    return ((bits << 4)  & 0xfff0) | ((bits >> 12) & 0x000f);

  // flip piece
  return bits & (0x4000 | 0x40) | // bit 14 and 6 remain the same
    ((bits >> 2) & 0x2000)      | // bit 15 becomes bit 13
    ((bits << 2) & 0x8000)      | // bit 13 becomes bit 15
    ((bits >> 12) & 0x01)       | // bit 12 becomes bit 0
    ((bits >> 10) & 0x02)       | // bit 11 becomes bit 1
    ((bits >> 8) & 0x04)        | // bit 10 becomes bit 2
    ((bits >> 6) & 0x08)        | // bit 9 becomes bit 3
    ((bits >> 4) & 0x10)        | // bit 8 becomes bit 4
    ((bits >> 2) & 0x20)        | // bit 7 becomes bit 5
    ((bits << 2) & 0x80)        | // bit 5 becomes bit 7
    ((bits << 4) & 0x0100)      | // bit 4 becomes bit 8
    ((bits << 6) & 0x0200)      | // bit 3 becomes bit 9
    ((bits << 8) & 0x0400)      | // bit 2 becomes bit 10
    ((bits << 10) & 0x0800)     | // bit 1 becomes bit 11
    ((bits << 12) & 0x1000);      // bit 0 becomes bit 12
}



void
find_equals()
{
  for (unsigned i = 0; i < navailable; i++)
    for (unsigned j = i+1; j < navailable; j++)
      for (unsigned l = 0; l < 8; l++)
        if (pieces_list[i].bit[0] == pieces_list[j].bit[l]) {
	  pieces_list[i].equals[pieces_list[i].nequals++] = j;
	  pieces_list[j].equals[pieces_list[j].nequals++] = i;

	  /* Is NR_EQUALS big enough? */
	  if ((pieces_list[i].nequals == NR_EQUALS-1) ||
	      (pieces_list[j].nequals == NR_EQUALS-1)) {
	    fprintf(stderr, "Warning: increase NR_EQUALS!\n");
	  }

          if (verbose >= 2)
            printf("%s is equal to %s in position %d\n",
                  pieces_list[i].name, pieces_list[j].name, l);

	  // Not necessary to check it for other positions of j
	  break;
	}
}


void
init()
{
  int i, j;

  for (i = 0; i < NR_WIRS; i++) {
    pieces_list[i].busy = false;
    pieces_list[i].nequals = 0;

    for (j = 0; j < 8; j++)
      pieces_list[i].symm[j] = 0;
    for (j = 0; j < NR_EQUALS; j++)
      pieces_list[i].equals[j] = -1;
  }

  for (i = 0; i < NR_WIRS; i++)
    for (j = 0; j < NBITS; j++)
      recipe[i][j].valid = false;
}

void
build(struct piece_t *structure[NR_WIRS], unsigned i)
{
  unsigned nhist = 0;
  unsigned history[NR_WIRS];

  // Finished?
  if (i == ncomplete) {

    nsolutions++;

    if (silence >= 3)
      return;

    // Report a solution
    if (docount)
      printf("Solution %d (found on try %lu)\n", nsolutions, tries);
    else
      printf("Solution %d\n", nsolutions);

    if (silence >= 2)
      return;

    for (unsigned j = 0; j < ncomplete; j++)
      printf("%2d-%s: %d.\n", j+1, structure[j]->name, structure[j]->pos+1);

    if (silence < 1) {
      printf("Hit enter to continue\n");
      getchar();
    }

    printf("\n");
    return;
  }


  // go through all pieces and try to fulfill all requirements by calling
  // function pointers. all should return true.
  for (unsigned j = 0; j < navailable; j++) {
    // Only change first piece when there are more pieces available than
    // needed. Otherwise this is the point where we quit.
    if (!i && j && !(navailable > ncomplete))
      return;

    piece_t *wd = &(pieces_list[j]);
    if (wd->busy)
      continue;

    // if an equally shaped piece has been tried already, skip this one
    if (wd->nequals) {
      unsigned useless = 0;
      for (unsigned j = 0; j < nhist; j++)
        for (unsigned k = 0; k < wd->nequals; k++)
          if (history[j] == (unsigned) wd->equals[k])
            useless = history[j];

      if (useless) {
        if (verbose >= 2)
          printf("No use to try %s in pos %d, because %s is the same!\n",
                wd->name, i, pieces_list[useless].name);
	continue;
      } else if (verbose >= 2)
	printf("Yes, equal to SOMETHING but first. Going ahead with %s in pos %d\n",
	   wd->name, i);
    }

    // remember that we tried piece j.
    history[nhist++] = j;

    // put piece in structure
    structure[i] = wd;

    // try piece in 8 different ways
    for (wd->pos = 0; wd->pos < 8; wd->pos++) {
      if (verbose >= 2)
        printf("Trying piece %s in pos %d, local pos %d\n", wd->name, i, wd->pos);

      if (wd->symm[wd->pos]) {
        if (verbose >= 2)
          printf("Skipping %s : %d\n", wd->name, wd->pos);
        continue;
      }

      if (docount) {
        tries++;
        if (counter == COUNT_STOP) {
          printf("Try %lu coming up...\n", tries);
          counter = 1;
        } else
          counter++;
      }

      // satisfy all set conditions
      bool fits = true;
      for (unsigned l = 0; l < NBITS; l++) {
        struct recipe_t *wl = &(recipe[i][l]);
        if (!wl->valid) 
          break;

        if (!(*wl->Fits) (wl->FitBits, structure)) {
          fits = false;
          if (verbose >= 2)
            printf("Fault at %d\n", l);
          break;
        }
      }

      if (!fits)
        continue;

      // put piece in structure and descend
      if (verbose >= 2)
        printf("*** Fits: %s, %d, %d ***\n", wd->name, i, wd->pos);
      wd->busy = true;
      build(structure, i+1);
      
      // don't try first piece in other positions
      if (!i)
        break;
    }

    // take piece out of structure
    wd->pos = 0;
    wd->busy = false;

    if (verbose >= 2)
      printf("Took out piece %s from position %d.\n", wd->name, i+1);
  }
}



int
SimpleFit(struct bit_t FitBits[BIT_SPECS], struct piece_t *structure[NR_WIRS])
{
  return BIT(0)+BIT(1) == 1;
}


int
OpenFit(struct bit_t FitBits[BIT_SPECS], struct piece_t *structure[NR_WIRS])
{
  /* Return true if either * sum in corner is 1 * sum in corner is 0 AND (one 
     of the bits next to corner is open) */
  register int sum;

  sum = BIT(0)+BIT(1);
  return (sum == 1 || ((!sum) && (!BIT(2) || !BIT(3))));
}

int
CornerFit(struct bit_t FitBits[BIT_SPECS], struct piece_t *structure[NR_WIRS])
{
  return (BIT(0)+BIT(1)+BIT(2) == 1);
}


int
QuadFit(struct bit_t FitBits[BIT_SPECS], struct piece_t *structure[NR_WIRS])
{
  return (BIT(0)+BIT(1)+BIT(2)+BIT(3) == 1);
}


int
DoubleFit(struct bit_t FitBits[BIT_SPECS], struct piece_t *structure[NR_WIRS])
{
  return (BIT(0)+BIT(1) <= 1);
}


int
TripleFit(struct bit_t FitBits[BIT_SPECS], struct piece_t *structure[NR_WIRS])
{
  /* Return true if three bits are together 1 OR * three bits are together 0
     and empty bit is not closed in. */
  register int sum;

  sum = BIT(0)+BIT(1)+BIT(2);
  return (sum == 1 || ((!sum) && (!BIT(3) || !BIT(4))));
}



int
read_pieces(char File[MAX_FILE_LEN])
{
  FILE *fp;

  /* Open file */
  if ((fp = fopen(File, "r")) == NULL) {
    fprintf(stderr, "Couldn't open file %s. Errno = %d.\n", File, errno);
    return 0;
  }

  // Read one line from file
  for (unsigned i = 0; i < NR_WIRS && !feof(fp); i++) {
    struct piece_t *wd = &(pieces_list[i]);

    // Read name
    int r = fscanf(fp, "%s", wd->name);
    if (feof(fp))
      return i;
    if (!r) {
      fprintf(stderr, "Couldn't read name in file\n");
      return 0;
    }

    // Read bits
    unsigned bit;
    for (unsigned j = 0; j < NBITS; j++) {
      if (!fscanf(fp, "%1u", &bit)) {
	fprintf(stderr, "Couldn't read bit(s) file\n");
	return 0;
      }
      wd->bit[0] |= (bit << j);
    }

    if (verbose >= 2) {
      printf("%s: ", wd->name);
      print_bits(wd->bit[0]);
      printf("\n");
    }

    // Generate all possible positions for this piece
    for (unsigned j = 1; j < 8; j++)
      wd->bit[j] = flip(wd->bit[j-1], j-1);

    // Start position
    wd->pos = 0;

    // Do a symmetry check
    for (unsigned j = 1; j < 8; j++) {
      for (signed k = j-1; k >= 0; k--) {
        if (wd->bit[j] == wd->bit[k]) {
          wd->symm[j] = 1;
          break;
        }
      }
    }

    navailable++;
  }

  return navailable;
}


int
read_recipe(char File[MAX_FILE_LEN])
{
  FILE *fp;

  /* Open file */
  if ((fp = fopen(File, "r")) == NULL) {
    fprintf(stderr, "Couldn't open file %s. Errno = %d.\n", File, errno);
    return 0;
  }

  /* Read first piece number */
  unsigned piece_no;
  if (!fscanf(fp, "%u:", &piece_no)) {
    fprintf(stderr, "Couldn't read piece number in make file\n");
    return 0;
  }
  ncomplete++;

  /* FOR every piece, read the description how to fit it in cube */
  for (unsigned i = 0; i < navailable && !feof(fp); i++) {
    if (verbose >= 2)
      printf("Reading piece %d\n", ncomplete);

    for (unsigned j = 0; j < NBITS; j++) {
      int r = fscanf(fp, " %u:", &piece_no);
      if (feof(fp))
	break;

      /* Isn't this start of new piece? Did the %d work? */
      if (r) {
	ncomplete++;
	break;
      }

      /* %d didn't work. It is a letter. Read it */
      char func_type;
      r = fscanf(fp, " %c:", &func_type);

      if (!r) {
	fprintf(stderr, "Couldn't read function type in make file: %d\n", r);
	return 0;
      }

      unsigned specs;
      switch (func_type) {
      case ('S'):
	recipe[i][j].Fits = SimpleFit;
	specs = 2;
	break;
      case ('O'):
	recipe[i][j].Fits = OpenFit;
	specs = 4;
	break;
      case ('C'):
	recipe[i][j].Fits = CornerFit;
	specs = 3;
	break;
      case ('Q'):
	recipe[i][j].Fits = QuadFit;
	specs = 4;
	break;
      case ('D'):
	recipe[i][j].Fits = DoubleFit;
	specs = 2;
	break;
      case ('T'):
	recipe[i][j].Fits = TripleFit;
	specs = 5;
	break;
      default:
        fprintf(stderr, "Unkown type %c\n", func_type);
        exit(-1);
      }

      /* specs says how many n.n we expect */
      for (unsigned k = 0; k < specs; k++) {
        unsigned p, b;
	if (!fscanf(fp, " %u.%u", &p, &b)) {
	  fprintf(stderr, "Couldn't read n.n from make file in piece %d\n",
		  ncomplete+1);
	  return 0;
	}

        if (verbose >= 2)
          printf("p = %d, b = %d\n", p, b);

	recipe[i][j].FitBits[k].piece = p-1;
	recipe[i][j].FitBits[k].bit = b-1;

	// read a "-" when necessary
	if (k < specs-1) {
          char dummy;
	  if (!fscanf(fp, " %c", &dummy)) {
	    fprintf(stderr, "Couldn't read from make file\n");
	    return 0;
	  }
	}
      }
      recipe[i][j].valid = true;
    }
  }

  return ncomplete;
}



int
main(int argc, char *argv[])
{
  struct piece_t *structure[NR_WIRS];
  bool readpieces = false;
  char recipe_file[MAX_FILE_LEN] = WIR_MAKE_FILE;
  int i;

  /* Parse arguments */
  for (i = 1; argv[i]; i++)
    if (!strcmp("-d", argv[i]) || !strcmp("--description", argv[i])) {
      if (!read_pieces(argv[++i])) {
        fprintf(stderr, "Couldn't read wirrel description file.\n");
        exit(-1);
      }
      readpieces = true;

    } else if (!strcmp("-m", argv[i]) || !strcmp("--recipe_file", argv[i]))
      strcpy(recipe_file, argv[++i]);
    else if (!strcmp("-s", argv[i]) || !strcmp("--silent", argv[i]))
      silence++;
    else if (!strcmp("-c", argv[i]) || !strcmp("--count", argv[i]))
      docount++;
    else if (!strcmp("-v", argv[i]) || !strcmp("--verbose", argv[i]))
      verbose++;
    else {
      usage(argv);
      exit(-1);
    }

  init();

  if(!readpieces) {
    if (!read_pieces(WIR_DES_FILE)) {
      fprintf(stderr, "Couldn't read wirrel description file.\n");
      exit(-1);
    }
  }

  // read recipe
  if (!read_recipe(recipe_file)) {
    fprintf(stderr, "Couldn't read wirrel make file.\n");
    exit(-1);
  }

  find_equals();

  if (verbose >= 1)
    printf("ncomplete = %d\n", ncomplete);

  // Are there enough pieces?
  if (ncomplete > navailable)
    fprintf(stderr, "Not enough pieces to make the object.\n");

  // Get the ball rolling
  build(structure, 0);

  return 0;
}
